/************************************************************************
 * NAME:	date.h
 *
 * DESCR:	Manipulates HDOS-style dates.
 *
 * NOTES:	
 ************************************************************************/

#include <time.h>
#include "hdos.h"

/************************************************************************
 * NAME:	hdos_date_pack()
 *
 * DESCR:	Pack the given HDOS date into an int.  Note that storage
 *		of an int is done in reverse order, but this routine
 *		does NOT do that.  Use "reverse_int()" on this routine
 *		to do that.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 *		0010110 0001 11011 = 27-Jan-92
 *		------- ---- -----
 *		year   mnth  day
 *		|      |    |
 *		|      |    +--- 27d
 *		|      +--------  1d (Jan=1, Feb=2, etc.)
 *		+--------------- 22d (70 + 22 = 92)
 *
 * 	In split octal, this date is 054.073A.
 *	HDOS will store the date in the normal low-order-byte first order
 *	used to store all 16-bit numbers, so this date would appear in the
 *	label or directory as 073 054.
 ************************************************************************/

#define YEARBITS	0xfe00
#define MONTHBITS	0x01e0
#define DAYBITS		0x001f

#define YEARSHIFT	9
#define MONTHSHIFT	5
#define DAYSHIFT	0

int
hdos_date_pack(struct hdos_date *d)
{
    int	r = 0;

    r |= ((d->year - 1970)<<YEARSHIFT) & YEARBITS;
    r |= (d->month << MONTHSHIFT) & MONTHBITS;
    r |= (d->day << DAYSHIFT) & DAYBITS;

    return(r);
}

struct hdos_date *
hdos_date_unpack(int r, struct hdos_date *d)
{
    d->year = ((r & YEARBITS) >> YEARSHIFT) + 1970;
    d->month = (r & MONTHBITS) >> MONTHSHIFT;
    d->day = (r & DAYBITS) >> DAYSHIFT;
}

/************************************************************************
 * NAME:	hdos_date_format()
 *
 * DESCR:	Format the given date into the buffer.
 *
 * ARGS:	format = 0  ===>  12/31/1998
 *		format = 1  ===>
 *
 * RETURNS:	returns the number of places consumed in buffer.
 *
 * NOTES:	
 ************************************************************************/
int
hdos_date_format(struct hdos_date *d, char *buffer, int format)
{
    char *ptr = buffer;

    switch(format) {
    case 0:
	buffer += sprintf(buffer,"%02d/%02d/%4d",d->month, d->day, d->year);
	break;
    }
    return(buffer-ptr);
}

/************************************************************************
 * NAME:	hdos_date_now()
 *
 * DESCR:	Returns an hdos_date that refers to right now.  The
 *		problem is that hdos_dates can only represent those dates
 *		from 1970 to 1999...so this thing will always return
 *		1999.
 *
 * ARGS:	
 *
 * RETURNS:	nothing
 *
 * NOTES:	- the structure is filled-in upon return.
 ************************************************************************/
void
hdos_date_now(struct hdos_date *now)
{
    struct tm	tm;
    time_t	t;

    time(&t);

    tm = *localtime(&t);

    now->year = tm.tm_year;
    now->month = tm.tm_mon;
    now->day = tm.tm_mday;

    /* bummer, but the following has to be done	*/

    now->year = 1999;
}


